--------------------------------------------------------------------------------
-- DIO2 board LCD driver
-- low level timing driver
--
-- Michal TRS
-- trsm1@fel.cvut.cz 
--------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;


entity lcd_time_drv is
  Generic (
   T     : integer := 20;
   tsu   : integer := 40;
   tw    : integer := 220;
   tc    : integer := 500);
  Port (
   CLK      : in std_logic;
   AS_RESET : in std_logic; 
   -- ---------------------
   DATA_IN     : in std_logic_vector(7 downto 0);
   RW_IN       : in std_logic;
   RS_IN       : in std_logic;
   DATA_VLD    : in std_logic;
   -- ------------------------
   DATA_OUT    : out std_logic_vector(7 downto 0);
   RW_OUT      : out std_logic;
   RS_OUT      : out std_logic;
   E           : out std_logic;
   -- -------------------------   
   RDY         : out std_logic);
end lcd_time_drv;


architecture lcd_time_drv_body of lcd_time_drv is

   -- input regs
   signal data_reg   : std_logic_vector(7 downto 0);
   signal rw_reg     : std_logic;
   signal rs_reg     : std_logic;
   
   signal start      : std_logic;

   -- 20 ns = 50 MHz , max delay 500 ns => 4 bit
   signal cnt        : std_logic_vector(3 downto 0);
   signal cnt_ce     : std_logic;
   signal cnt_load   : std_logic;
   signal cnt_data   : std_logic_vector(3 downto 0);
   signal cnt_zero   : std_logic; 
   
   type t_ctrl_state is (st_rdy, st_wtsu, st_wtw, st_wtc);
   signal cur_state  : t_ctrl_state;
   signal next_state : t_ctrl_state;

begin

   -- ---------------------------------------
   -- input signals registred
   -- ---------------------------------------

   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         data_reg <= (others => '0');
      elsif CLK = '1' and CLK'event then
         if DATA_VLD = '1' then
            data_reg <= DATA_IN;
         end if;
      end if;
   end process;

   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         rw_reg <= '0';
      elsif CLK = '1' and CLK'event then
         if DATA_VLD = '1' then
            rw_reg <= RW_IN;
         end if;
      end if;
   end process;

   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         rs_reg <= '0';
      elsif CLK = '1' and CLK'event then
         if DATA_VLD = '1' then
            rs_reg <= RS_IN;
         end if;
      end if;
   end process;


   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         start <= '0';
      elsif CLK = '1' and CLK'event then
         start <= DATA_VLD;
      end if;      
   end process;

   -- ----------------------------------------
   -- driving FSM
   -- ----------------------------------------
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         cur_state <= st_rdy;
      elsif CLK = '1' and CLK'event then
         cur_state <= next_state;
      end if;
   end process;

   -- next state logic
   process(cur_state, start, cnt_zero)
   begin
      next_state <= cur_state;

      case cur_state is
         when st_rdy =>
            if start = '1' then
               next_state <= st_wtsu;
            end if;
         when st_wtsu =>
            if cnt_zero = '1' then
               next_state <= st_wtw;
            end if;
         when st_wtw =>
            if cnt_zero = '1' then
               next_state <= st_wtc;
            end if;
         when st_wtc =>
            if cnt_zero = '1' then
               next_state <= st_rdy;
            end if;
      end case;
   end process;

   -- output logic
   process(cur_state, start, cnt_zero)
   begin                
      cnt_data <= (others => '0'); 
      cnt_load <= '0';
      cnt_ce   <= '0';
      E        <= '0';
      RDY      <= '0';
   
      case cur_state is

         when st_rdy =>
            if start = '0' then
               RDY <= '1';
            else
               cnt_data <= conv_std_logic_vector(tsu / T,4);
               cnt_load <= '1';
               cnt_ce   <= '1';
            end if;

         when st_wtsu =>
            cnt_ce   <= '1';

            if cnt_zero = '1' then
               cnt_data <= conv_std_logic_vector(tw / T,4);
               cnt_load <= '1';
               E        <= '1';
            end if;                  

         when st_wtw =>
            cnt_ce   <= '1';
            E        <= '1';

            if cnt_zero = '1' then
               cnt_data <= conv_std_logic_vector((tc - tsu - tw) / T,4);
               cnt_load <= '1';
            end if;                  

         when st_wtc =>
            cnt_ce   <= '1';

      end case;
   end process;

   -- -------------------------------- 
   -- counter
   -- --------------------------------

   icnt:process(CLK,cnt_ce,cnt_load)
   begin
      if CLK = '1' and CLK'event then
         if cnt_load = '1' then
            cnt <= cnt_data; 
         elsif cnt_ce = '1' then
            cnt <= cnt - 1;  
         end if;    
      end if;
   end process;
   
   cnt_zero <= '1' when cnt = conv_std_logic_vector(0,4)
          else '0'; 

   -- -------------------------------
   -- OUTPUT signal connection
   -- -------------------------------

   DATA_OUT <= data_reg when (cur_state /= st_rdy)
      else (others => 'Z');
   RW_OUT   <= rw_reg;
   RS_OUT   <= rs_reg;
end lcd_time_drv_body;
