/*
  hci.h

  host controller interface routines
*/
/* 
   BlueMP3 firmware (c) 2004 by Till Harbaum, harbaum@beecon.de
*/


#ifndef HCI_H
#define HCI_H

#include "types.h"
#include "hci_con.h"

// #define DEVICE_CLASS  (COD_Major_Peripheral)
//#define DEVICE_CLASS  (COD_Audio | COD_Major_Audio)
#define DEVICE_CLASS  (COD_Information | COD_Toy)


#define DEVICE_NAME   "NutOS-BT"


/* HCI error codes */
#define HCI_ERROR_AUTH_FAILURE      0x05
#define HCI_ERROR_REJECT_LIMIT      0x0d
#define HCI_ERROR_REJECT_SECURITY   0x0e
#define HCI_ERROR_REJECT_PDEV       0x0f

/* MicroBlue internal error codes */
#define HCI_ERROR_MB_NO_SLOT        0x40



// hci_packet_type_constants
#define HCI_CMD 1
#define HCI_ACL 2
#define HCI_SCO 3
#define HCI_EVT 4

/*
  Opcodes consist of an 10-bit OCF and a 6-bit OGF part. The following
  macros transform (OGF,OCF) pairs into (LOW, HIGH) pairs and back.
  Since Bluetooth does currently not define any OCF greater than 0x3f we
  can simplify the OCF calculations for decoding. (For encoding the
  macros are executed at compile-time. Hence optimization is
  unneccessary.)
*/

/* create a opcode in two bytes */
#define OPCODE(ogf, ocf) (ocf & 0xff), (((ocf>>8)&3) | ((ogf<<2)&0xff))
/* create a opcode in one word */
#define OPCODE16(ogf, ocf) (((ogf) << 10) | (ocf))

#define OCF(low, high) (low & ((high&3)<<8))
#define OGF(low, high) (high>>2)

#undef OCF
#define OCF(low, high) (low) 

/* HCI COMMANDS */
#define OGF_LINK_CTL               0x01
#define OGF_LINK_POLICY            0x02
#define OGF_HOST_CTL               0x03
#define OGF_INFO_PARAM             0x04
#define OGF_STATUS_PARAM           0x05
#define OGF_VENDOR                 0x3f

/* OGF_LINK_CTL */
#define OCF_ACCEPT_CONN_REQ        0x09
#define OCF_REJECT_CONN_REQ        0x0a
#define OCF_PIN_CODE_REQUEST_REPLY 0x0d
#define OCF_PIN_CODE_NEGATIVE_REPLY 0x0e
#define OCF_LINK_KEY_NEGATIVE_REPLY 0x0c


/* OGF_HOST_CTL */
#define OCF_RESET                  0x03
#define OCF_CHANGE_LOCAL_NAME      0x13
#define OCF_WRITE_SCAN_ENABLE      0x1a
#define OCF_WRITE_COD              0x24



/* OGF_INFO_PARAM */
#define OCF_READ_BUFFER_SIZE       0x05

#define INQUIRY_SCAN_ENABLE        0x01
#define PAGE_SCAN_ENABLE           0x02

#define LINK_TYPE_SCO              0x00
#define LINK_TYPE_ACL              0x01

/* flow control values */
#define HCI_FLOW_ACL               0x01
#define HCI_FLOW_SCO               0x02

/*connection reject reasons */
#define LIMITED_RESOURCES          0x0d

/* COD major service class */
#define COD_Audio                    0x00200000L
#define COD_Major_Audio              0x00000400L
#define COD_Information              0x00800000L
#define COD_Toy              		 0x00000114L



/* convert integer into two/three byte sequence */
#define MK_SEQ2(a)  (((u16_t)a)&  0xff),((((u16_t)a)>>8)&  0xff)
#define MK_SEQ3(a)  (((u32_t)a)&  0xff),((((u32_t)a)>>8)&  0xff),((((u32_t)a)>>16)&  0xff)

#define HCI_DECODE_CONT 0
#define HCI_DECODE_OK   1
#define HCI_DECODE_ERR  2

/* reasons for hci decoding, used to avoid recursion */
#define HCI_PROCESS_IDLE        0   // ordinary processing
#define HCI_PROCESS_WAIT_ACL    1   // waiting for free acl slot
#define HCI_PROCESS_WAIT_CMD    2   // waiting for free cmd buffers

/* hci command parameter flags */
#define HCI_RAM          0x0000
#define HCI_ZERO         0xc000
#define HCI_FLAG_MASK    0xc000
#define HCI_LEN_MASK     0x3fff

void   hci_init(void);
void   hci_process(u16_t cmd);
void   hci_send_seq(u08_t *data, u08_t len);
void   hci_set_name(void);

#define HCI_PARMS   0x80  // flag indicating that a parameter pointer is present
void   hci_send_cmd(u08_t *, u08_t, u08_t, ...);

bool_t hci_reset(void);

#endif // HCI_H
