/*
  l2cap.c

  logical link control and adaption protocol (l2cap)

*/
/* 
   BlueMP3 firmware (c) 2004 by Till Harbaum, harbaum@beecon.de
*/




#include "types.h"
#include "hci.h"
#include "hci_uart.h"
#include "hci_event.h"
#include "hci_info.h"
#include "hci_acl.h"
#include "hci_con.h"
#include "hci_callback.h"
#include "l2cap.h"


/* send a l2cap command */
static void l2cap_send_cmd(hci_acl_con_t *acl, u08_t code, 
			   u08_t id, u08_t slot,
			   u16_t len) {
  hci_acl_send(acl, len+8, TRUE);   // acl header
  
  hci_uart_put_u16(len+4);          // l2cap length
  hci_uart_put_u16(1);              // l2cap channel

  hci_uart_put(code);

  /* id==0 means, command (not response) -> so send local ident field */
  /* otherwise use ident field from request */
  if(!id) {
    acl->ident++;
    if(!acl->ident) acl->ident++;    /* skip reserved zero id */
    hci_uart_put(acl->ident);        /* send and increase id */

    /* save if in the current slot */
    acl->l2con[slot].lid = acl->ident;
  } else 
    hci_uart_put(id);

  hci_uart_put_u16(len);            // payload length
}

/* send a l2cap configuration request */
void l2cap_send_configuration_request(hci_acl_con_t *acl, 
				      u08_t slot, u08_t id) {

  l2cap_send_cmd(acl, L2CAP_CONF_REQ, 0, slot, 4);
  
  hci_uart_put_u16(acl->l2con[slot].scid);
  hci_uart_put_u16(0);  // flags
}

/* send l2cap connection request */
void l2cap_connection_request(hci_acl_con_t *acl, u08_t id) {
  u08_t slot = hci_con_new_l2cap_entry(acl);
  u16_t tmp_psm;

  if(slot != HCI_NO_SLOT) {
    /* ok, got a free slot, add "unconfigured" flags to psm */
    acl->l2con[slot].psm     = hci_acl_get_u16() | 0xc000;
    acl->l2con[slot].scid    = hci_acl_get_u16();

    /* setup cyclic entry for fast access to parent acl entry */
    acl->l2con[slot].acl     = (struct hci_acl_con*)acl;

    /* send connection response */
    l2cap_send_cmd(acl, L2CAP_CON_RSP, id, 0, 8);

    hci_uart_put_u16(0x40 + slot);
    hci_uart_put_u16(acl->l2con[slot].scid);

    /* remove flags from psm while doing application callback */
    tmp_psm = acl->l2con[slot].psm;
    acl->l2con[slot].psm &= 0x3fff;

    /* get ok from upper layers */
    if(hci_internal_callback(L2CAP_EVT_ACKNOWLEDGE_CONNECTION, 
			     &acl->l2con[slot])) {
      hci_uart_put_u16(L2CAP_CON_RSP_SUCCESS);

      /* restore psm */
      acl->l2con[slot].psm = tmp_psm;
    } else {
      hci_uart_put_u16(L2CAP_CON_RSP_ILLPSM);
      acl->l2con[slot].psm = 0;
    }
  } else {
    u16_t scid;

    /* no slot, error */
    hci_acl_get_u16();         // skip psm
    scid = hci_acl_get_u16();  // save scid

    /* send connection response */
    l2cap_send_cmd(acl, L2CAP_CON_RSP, id, 0, 8);
    hci_uart_put_u16(0);
    hci_uart_put_u16(scid);    // echo scid
    hci_uart_put_u16(L2CAP_CON_RSP_RESOURCE);
  }

  hci_uart_put_u16(0); // result code (only used for pending)

  /* send configuration request if connection was accepted */  
  if(acl->l2con[slot].psm)
    l2cap_send_configuration_request(acl, slot, id+1);
}

void l2cap_configuration_request(hci_acl_con_t *acl, u08_t id, u16_t len) {
  u16_t flags, slot;
  u08_t i, options[len-4];  // buffer for options (gcc special)

  /* read scid and flags */
  slot  = hci_acl_get_u16() - 0x40;
  flags = hci_acl_get_u16();
  len -= 4;

  /* read options into buffer */
  if(len) 
    for(i=0;i<len;i++)
      options[i] = hci_acl_get();

  /* send response (echoing of options removed) */
  l2cap_send_cmd(acl, L2CAP_CONF_RSP, id, 0, 6);
  hci_uart_put_u16(acl->l2con[slot].scid);
  hci_uart_put_u16(0);       // send result code
  hci_uart_put_u16(flags);   // echo flags

  /* remember, that we sent our response */
  acl->l2con[slot].psm &= ~0x8000;

  /* this BlueMP3 demo code does not verify the l2cap mtu size, since */
  /* the mp3 player will work with any mtu size the other size may request */

  /* if both configuration responses have been sent, notify */
  /* application */
  if(!(acl->l2con[slot].psm & 0xc000))
    hci_internal_callback(L2CAP_EVT_NOTIFY_CONNECTION, &acl->l2con[slot]);
}

void l2cap_disconnection_request(hci_acl_con_t *acl, u08_t id) {
  u08_t slot = hci_acl_get_u16() - 0x40;

  hci_acl_get_u16();
  
  /* send response */
  l2cap_send_cmd(acl, L2CAP_DISC_RSP, id, 0, 4);

  hci_uart_put_u16(slot + 0x40);
  hci_uart_put_u16(acl->l2con[slot].scid);

  hci_internal_callback(L2CAP_EVT_NOTIFY_DISCONNECTION, &acl->l2con[slot]);

  acl->l2con[slot].psm = 0;
}

void l2cap_configuration_response(hci_acl_con_t *acl, u08_t id, u16_t len) {
  u08_t slot;
  u16_t result;

  slot = hci_acl_get_u16() - 0x40;
  hci_acl_get_u16();       // ignore flags
  result = hci_acl_get_u16();

  /* drop rest of configuration data */
  if(len > 6)
    hci_acl_drop(len-6);

  if(id != acl->l2con[slot].lid) return;

  /* remember, that we received a configuration response */
  acl->l2con[slot].psm &= ~0x4000;

  /* if both configuration responses have been sent, notify */
  /* application */
  if(!(acl->l2con[slot].psm & 0xc000))
    hci_internal_callback(L2CAP_EVT_NOTIFY_CONNECTION, &acl->l2con[slot]);
}

void l2cap_link_broken(hci_acl_con_t *acl) {
  u08_t slot;
  
  /* remove all existing l2cap connections */
  for(slot=0;slot<MAX_L2CON;slot++) {
    if(acl->l2con[slot].psm)
      hci_internal_callback(L2CAP_EVT_NOTIFY_DISCONNECTION, &acl->l2con[slot]);

    acl->l2con[slot].psm = 0;
  }
}

/* echo number of l2cap bytes */
void l2cap_echo(hci_acl_con_t *acl, u08_t id, u16_t len) {
  u08_t buffer[len];
  u16_t i;
  
  /* fill buffer */
  for(i=0;i<len;i++) buffer[i] = hci_acl_get();

  /* send reply */
  l2cap_send_cmd(acl, L2CAP_ECHO_RSP, id, 0, len);
  hci_uart_send(buffer, len);
}

void l2cap_decode(hci_acl_con_t *acl, u16_t acl_len) {
  u16_t channel;

  acl_expect = acl_len;

  /* total payload to expect is length */
  acl->l2cap_expect = hci_acl_get_u16();
  channel = hci_acl_get_u16();

  /* l2cap command channel? */
  if(channel == 1) {
    /* get command header */
    u08_t code = hci_acl_get();
    u08_t id   = hci_acl_get();
    u16_t len  = hci_acl_get_u16();

    /* verify if header length makes sense */
    if(len+4 != acl->l2cap_expect) {
      /* ignore everything */
      hci_acl_drop(acl->l2cap_expect);
      return;
    }

    switch(code) {
      case L2CAP_ECHO_REQ:
	/* echo payload */
	l2cap_echo(acl, id, len);
	break;

      case L2CAP_CON_REQ:
	l2cap_connection_request(acl, id);
	break;

      case L2CAP_CONF_REQ:
	l2cap_configuration_request(acl, id, len);
	break;

      case L2CAP_DISC_REQ:
	l2cap_disconnection_request(acl, id);
	break;

      case L2CAP_CONF_RSP:
	l2cap_configuration_response(acl, id, len);
	break;

      default:
	/* just ignore unsupported commands in this BlueMP3 demo code */
	/* the full MicroBlue supports all these commands */
	hci_acl_drop(len);
	break;
    }
  } else {
    /* notify application about availability of data */
    hci_internal_callback(L2CAP_EVT_NOTIFY_RECEIVE, &acl->l2con[channel-0x40]);

    /* still l2cap data to be expected? drop it! */
    if(acl->l2cap_expect) {
      l2cap_receive(&acl->l2con[channel-0x40], NULL, acl->l2cap_expect);
    }
  }
}

void l2cap_receive(hci_l2con_t *l2con, u08_t *buffer, u16_t len) {
  u16_t bytes2copy;
  hci_acl_con_t *acl = (hci_acl_con_t*)l2con->acl;

  /* make sure the user is not too greedy */
  if(len > acl->l2cap_expect) len = acl->l2cap_expect;

  /* fetch as many bytes as the user wants */
  while(len) {
    /* refill acl buffer if required */
    while(!acl_expect) hci_process(HCI_PROCESS_IDLE);

    /* determine buffer size */
    bytes2copy = len;
    if(acl_expect        < bytes2copy) bytes2copy = acl_expect;
    if(acl->l2cap_expect < bytes2copy) bytes2copy = acl->l2cap_expect;

    /* reduce counters */
    len               -= bytes2copy;
    acl_expect        -= bytes2copy;
    acl->l2cap_expect -= bytes2copy;

    if(buffer) {
      /* fetch all bytes */
      while(bytes2copy--)
	*buffer++ = hci_uart_get();
    } else
      hci_uart_drop(bytes2copy);
  }
}

/* return a single byte */
u08_t l2cap_receive_u08(hci_l2con_t *l2con) {
  u08_t val;

  l2cap_receive(l2con, (u08_t*)&val, sizeof(val));

  return val;
}

void l2cap_send(hci_l2con_t *l2con, void *data, u16_t len) {
  u08_t *data8 = (u08_t*)data;

  // first fragment contains the the 4 bytes l2cap header
  u16_t acl_len = hci_acl_fsize() - 4;
  hci_acl_con_t *acl = (hci_acl_con_t*)l2con->acl;

  /* init fragment counters */
  acl2send = acl->l2cap2send = len;

  /* limit size of acl chunk */
  if(acl2send > acl_len) acl2send = acl_len;

  /* acl header */
  hci_acl_send(acl, acl2send + 4, TRUE);

  /* l2cap header */
  hci_uart_put_u16(len);
  hci_uart_put_u16(l2con->scid);

  /* still more data than fits into the acl packet? */
  while(len > acl2send) {
    hci_uart_send(data8, acl2send);
    len               -= acl2send;
    acl->l2cap2send   -= acl2send;
    data8             += acl2send;

    /* determine size of next acl chunk */
    acl2send           = acl->l2cap2send;

    /* limit size of acl chunk */
    if(acl2send > hci_acl_fsize()) 
      acl2send = hci_acl_fsize();

    /* send acl header */
    hci_acl_send(acl, acl2send, FALSE);
  }

  /* send remaining data */
  if(len > 0) {
    hci_uart_send(data8, len);
    acl->l2cap2send -= len;
    acl2send        -= len;
  }
}
