/*
    l2send, L2CAP sender application for Linux/BlueZ

    Copyright (C) 2004 Till Harbaum <till@harbaum.org>,
	              2007 Martin Tuma <tumam3@fel.cvut.cz>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>

#include <netinet/ip.h>
#include <netinet/udp.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>

#include <errno.h>

/* stuff from bluetooth.h, which interferes with other includes */
#include <bluetooth/bluetooth.h>
#include <bluetooth/l2cap.h>

#define APP_PSM 0x1023

int buffer_size = 64;

int bluez_open(char *bdaddr_str, int hci_idx) {
  struct sockaddr_l2 rem_addr, loc_addr;
  struct l2cap_options opts;
  int s, opt;

  /* create a bluetooth socket */
  if( (s = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP)) < 0 ) {
    printf("Can't create socket. %s(%d)\n", strerror(errno), errno);
    exit(1);
  }

  /* setup control structures */
  memset(&loc_addr, 0, sizeof(loc_addr));
  loc_addr.l2_family = AF_BLUETOOTH;
  loc_addr.l2_psm    = htobs(APP_PSM);
  if( bind(s, (struct sockaddr *) &loc_addr, sizeof(loc_addr)) < 0 ) {
    printf("Can't bind socket. %s(%d)\n", strerror(errno), errno);
    exit(1);
  }

  memset(&rem_addr, 0, sizeof(rem_addr));
  rem_addr.l2_family = AF_BLUETOOTH;
  baswap(&rem_addr.l2_bdaddr, strtoba(bdaddr_str));
  rem_addr.l2_psm = htobs(APP_PSM);
  if( connect(s, (struct sockaddr *)&rem_addr, sizeof(rem_addr)) < 0 ){
    printf("Can't connect. %s(%d)\n", strerror(errno), errno);
    close(s);
    return -1;
  }
  
  /*
  opt = sizeof(opts);
  if( getsockopt(s, SOL_L2CAP, L2CAP_OPTIONS, &opts, &opt) < 0 ){
    printf("Can't get L2CAP options. %s(%d)\n", strerror(errno), errno);
    close(s);
    return -1;
  }
  
  // use omtu for output buffer size
  buffer_size = opts.omtu;
  */

  return s;
}


void send_file(int s, char *name, int chunk_size) {
  FILE *file;
  char buffer[chunk_size];
  int bytes2send;


  if(!(file = fopen(name, "rb"))) {
    fprintf(stderr, "Error opening file \"%s\": %s\n", name, strerror(errno));
    return;
  }

  do {
    if(!(bytes2send = fread(buffer, 1, sizeof(buffer), file))) {
      if(!feof(file)) {
        fprintf(stderr, "Error reading file \"%s\": %s\n", 
         name, strerror(errno));
		return;
	  }
	} else {
	  if(write(s, buffer, bytes2send) != bytes2send) {
	    fprintf(stderr, "Error sending: %s\n", strerror(errno));
        return;
	  }
    }       
  } while(bytes2send);
    
  fclose(file);
}


void send_message(int s, char* msg, int len) {
  if(write(s, msg, len) != len)
    fprintf(stderr, "Error sending: %s\n", strerror(errno));
}


void usage(void) {
  printf("Usage: l2send bdaddr file\n");
  exit(1);
}


int main(int argc, char **argv) {
  int s, hci=0;
  char buffer[buffer_size];
  int rec_bytes;


  if(argc < 3 ) usage();

  /* open device if the first file entry is found */
  if((s = bluez_open(argv[1], hci)) < 0) {
    fprintf(stderr, "Unable to connect to %s.\n", argv[1]);
	return -1;
  }
  printf("Connected\n");
  
  while(1) {

	if ((rec_bytes = read(s, buffer, sizeof(buffer))) > 0) {
	  buffer[rec_bytes] = '\0';
	  printf(">%s", buffer);
	    
	  if (!strncmp(buffer, "file", 4))
        send_file(s, argv[2], buffer_size);
	  else if (!strncmp(buffer, "exit", 4)) {
	    send_message(s, ">Sayonara\n", 10);
	    break;
	  }else
	    send_message(s, ">Unknown command\n", 17);
	} else
	  if (rec_bytes == -1) {
        fprintf(stderr, "Error receiving: %s\n", strerror(errno));
		return -1;
	  }  	    
  }

  return 0;
}
