/* analyzer.cpp 
 * Analyse TCP/IP (FTP session)
 * 
 * Michal Trs, Jirka Sejtko
 * CVUT FEL, K336
 */

/* https://dsn.felk.cvut.cz/wiki/doku.php?id=vyuka:cviceni:x36los:uloha1-zadani	*
 * http://en.wikipedia.org/wiki/Transmission_Control_Protocol					*/


#include <stdio.h>
#include <pcap.h>
#include <string.h>
#include "types.h"


// filter
char filter_exp[FILTER_BUF_SIZE];	/* filter expression buffer */
bpf_u_int32 NetMask = 0xffffff;		/* IP Mask network */
struct bpf_program fcode;			/* compiled filter */

// offline file
pcap_t *fp;
char errbuf[PCAP_ERRBUF_SIZE];

// output files
FILE *fi, *fd; // fi - info file, fd - data file

// TCP state
tcp_state tcps = TCS_CLOSED;


// TCP window buffers
tcp_packet query[NUM_PACKETS];					/* client->server packet buffer */
tcp_packet response[NUM_PACKETS];				/* server->client packet buffer */
int qsize = 0, rsize = 0; 

// Call back function
void dispatcher_handler(u_char *, const struct pcap_pkthdr *, const u_char *);


/* 
 * compare 2 packets based on seq. numbers (qsort() cmp. fun.) (by Martin Tuma)
 */
int pkt_cmp(const void *a, const void *b)
{
	tcp_packet *pa, *pb;
	
	pa = (tcp_packet*) a; 
	pb = (tcp_packet*) b;

	if (pa->seq == pb->seq) return 0;
	return (pa->seq < pb->seq) ? -1 : 1; 
}


/*
 * compare 2 timeval structs (by Martin Tuma)
 */
int timeval_cmp (struct timeval a, struct timeval b)
{
	if (a.tv_sec > b.tv_sec) return 1;
	if (a.tv_sec < b.tv_sec) return -1;
	if (a.tv_usec > b.tv_usec) return 1;
	if (a.tv_usec < b.tv_usec) return -1;
	return 0;
}

/*
 * Save TCP data to file (unsafe - only for ASCIIZ protocol like FTP)
 */
void dump_tcp(FILE *f) {
		int i,qp=0,rp=0;

	for (i=0; i<qsize+rsize; i++) {
		if (qp < qsize && rp < rsize) {
			if (timeval_cmp(query[qp].ts, response[rp].ts) < 0) {
				fprintf(f,"\">>\",\"%s\"\n", query[qp].data);			
				qp++;
			} else {
				fprintf(f,"\"<<\",\"%s\"\n", response[rp].data);
				rp++;
			}
		} else if (qp == qsize) {
			fprintf(f,"\"<<\",\"%s\"\n", response[rp].data);
			rp++;
		} else if (rp == rsize) {
			fprintf(f,"\">>\",\"%s\"\n", query[qp].data);
			qp++;
		}
	}
}



/*
 * main function
 */
int main(int argc, char **argv)
{		
	if(argc != 6)
	{	
		printf("usage: %s wireshark_dump.pcap SRC_IP SRC_PORT DST_IP DST_PORT", argv[0]);
		return -1;
	}
	
	// open outputs file
	if ((fi = fopen("tcp_info.txt","wt")) == NULL) 
	{
		printf("Unable to create output info file");
		return -2;
	}


	printf("Processing TCP stream...\n");

	/* Open the capture file */
	if ((fp = pcap_open_offline(argv[1], errbuf)) == NULL)
	{
		fprintf(stderr,"\nUnable to open the file %s.\n", argv[1]);
		return -1;
	}
	

	/* make filter expression */
	snprintf(filter_exp, FILTER_BUF_SIZE,
		    "(src host %s and src port %s and dst host %s and dst port %s) or (src host %s and src port %s and dst host %s and dst port %s)",
		    argv[2], argv[3], argv[4], argv[5], argv[4], argv[5], argv[2], argv[3]);
	

	/* compile the filter */
	if(pcap_compile(fp, &fcode, filter_exp, 1, NetMask) < 0)
	{
		fprintf(stderr,"\nError compiling filter: wrong syntax.\n");

		pcap_close(fp);
		return -3;
	}

	/* set the filter */
	if(pcap_setfilter(fp, &fcode)<0)
	{
		fprintf(stderr,"\nError setting the filter\n");
		pcap_close(fp);
		return -4;
	}


	/* read and dispatch packets until EOF is reached */
	pcap_loop(fp, 0, dispatcher_handler, NULL);


	/* sort data buffers */
	qsort(query, qsize, sizeof(tcp_packet), pkt_cmp);
	qsort(response, rsize, sizeof(tcp_packet), pkt_cmp);


	/* dump TCP data */
	if ((fd = fopen("tcp_data.csv","wt")) == NULL) 
	{
		printf("Unable to create output data file");
		return -3;
	}
	
	dump_tcp(fd);



	fclose(fi);
	fclose(fd);
	pcap_close(fp);
	
	printf("DONE, press <ENTER> to exit...\n");
	getchar();
	return 0;
}


/* 
 * process TCP packet
 */
void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data)
{
    ip_header *ih;
	tcp_header *th;

	u_int ip_len;
	u_char tcp_len;
	unsigned tcp_data_len;
    u_short sport,dport;

	char src_ip[IP_ADDR_LEN], dst_ip[IP_ADDR_LEN];
    
	
    /* retireve IP and TCP headers and lenght*/
    ih = (ip_header *) (pkt_data + ETH_HDR_LEN);
    ip_len = (ih->ver_ihl & 0xf) * 4;
    th = (tcp_header *) ((u_char*)ih + ip_len);
	tcp_len = (th->data_ofs_res) / 4;  // 4-bit field specifies TCP header size in 32-bit words


	/* get IP FLAGS */
	u_char ip_flags = (u_char) ((ntohs(ih->flags_fo) & 0xE000) >> 12);


    /* get TCP ports */
    sport = ntohs( th->sport );
    dport = ntohs( th->dport );


	/* get TCP data offset and size */
	u_char *tcp_data = ((u_char*)th + 20);
	tcp_data_len = header->len - ETH_HDR_LEN - ip_len - tcp_len;
	tcp_data[tcp_data_len-2] = 0; // terminate string (each FTP message terminate with EOLN)


	/* build IP address in string format */
	snprintf(src_ip, FILTER_BUF_SIZE,
		    "%d.%d.%d.%d:%d",
		    ih->saddr.byte1, ih->saddr.byte2, ih->saddr.byte3, ih->saddr.byte4, sport
	);

	snprintf(dst_ip, FILTER_BUF_SIZE,
		    "%d.%d.%d.%d:%d",
		    ih->daddr.byte1, ih->daddr.byte2, ih->daddr.byte3, ih->daddr.byte4, dport
	);



	/********* TCP info file **************/
	
	/* IP address */
	fprintf(fi,"%s->%s\n", src_ip, dst_ip);


	/* IP flags */
	if (ip_flags & 0x4) fprintf(fi, "IP: Don't framgment: Set\n");
	if (ip_flags & 0x2) fprintf(fi, "IP: packet framgmented, offset : %u, data may be corrupted\n", ih->flags_fo && 0xFFFF); 


	/* print TCP flags */
	fprintf(fi,"Flags:");
	if (cwr(th->flags)) fprintf(fi," CWR");
	if (ece(th->flags)) fprintf(fi," ECE");
	if (urg(th->flags)) fprintf(fi," URG");
	if (ack(th->flags)) fprintf(fi," ACK");
	if (psh(th->flags)) fprintf(fi," PSH");	
	if (fin(th->flags)) fprintf(fi," FIN");	
	if (rst(th->flags)) fprintf(fi," RST");
	if (syn(th->flags)) fprintf(fi," SYN");	
	fprintf(fi,"\n");


	/* seq / ack nr */
	fprintf(fi, "Seq / Ack number: %u / %u\n", ntohl(th->seqnum), ntohl(th->acknum));


	/* MSS */
	if ((syn(th->flags)) && tcp_data_len == 0) {
		int opt_offset = 0;

		while(tcp_len > 20 + opt_offset) {
			tcp_options *tcp_opt = (tcp_options *) ((u_char*)th + 20 + opt_offset);		
			if ((tcp_opt->lenght == 4) && (tcp_opt->kind == 2)) {
				fprintf(fi,"MSS: %u B\n",ntohs(tcp_opt->mss));
				break;
			} else {
				if (tcp_opt->kind == 1) {
					opt_offset++;
				} else {
					opt_offset += tcp_opt->lenght;
				}
			}
		} // while
	}


	/* Window size */
	if (!psh(th->flags) && ack(th->flags) && tcp_data_len == 0) {
		fprintf(fi,"Window size: %u\n",ntohs(th->window));		
	}


	/* detect duplicate & switched packet order (code by Martin Tuma)*/
	if (tcp_data_len > 0) { // only for data packet
		if (dport == L5_PROTOCOL) {
			for (int i=0; i< qsize; i++) {
				if (query[i].seq == htonl(th->seqnum)) {
					printf("TCP: Packet retransmission!\n");
					return;
				}
			}
			query[qsize].ts = header->ts;
			query[qsize].seq = htonl(th->seqnum);
			query[qsize].len = tcp_data_len;
			memcpy(query[qsize].data, tcp_data, tcp_data_len);
			qsize++;
		} else if (sport == L5_PROTOCOL) {
			for (int i=0; i< rsize; i++) {
				if (response[i].seq == htonl(th->seqnum)) {
					printf("TCP: Packet retransmission!\n");
					return;
				}
			}
			response[rsize].ts = header->ts;
			response[rsize].seq = htonl(th->seqnum);
			response[rsize].len = tcp_data_len;
			memcpy(response[rsize].data, tcp_data, tcp_data_len);
			rsize++;
		}
	}
	

	/* Open / Close connection */
	switch (tcps) {
		case TCS_CLOSED:
			if (syn(th->flags)) {
				fprintf(fi,"Client: Connecting to server...\n");
				tcps = TCS_SYN_SENT;
			}
		case TCS_SYN_SENT: 
			if (syn(th->flags) && ack(th->flags)) {
				fprintf(fi,"Server: Opening connection...\n");
				tcps = TCS_SYN_RCVD;
			}
			break;
		case TCS_SYN_RCVD:
			if (ack(th->flags)) {
				fprintf(fi,"Client: Connection established\n");
				tcps = TCS_ESTABILISHED;
			}
			break;
		case TCS_ESTABILISHED:
			if (fin(th->flags)) {
				fprintf(fi,"Client: active close\n");
				tcps = TCS_FIN_WAIT1;
			}
			if (rst(th->flags) && sport == L5_PROTOCOL) {
				fprintf(fi,"Client: Connection reset by server\n");
				tcps = TCS_CLOSED;
			}
			if (rst(th->flags) && dport == L5_PROTOCOL) {
				fprintf(fi,"Server: Connection reset by peer\n");
				tcps = TCS_CLOSED;
			}
			break;
		case TCS_FIN_WAIT1:
			if (ack(th->flags)) {
				fprintf(fi,"Server: closing connection...\n");
				tcps = TCS_CLOSE_WAIT;
			}
			break;
		case TCS_CLOSE_WAIT:
			if (fin(th->flags)) {
				fprintf(fi,"Client: disconnected\n");
				tcps = TCS_LAST_ACK;
			}
			break;
		case TCS_LAST_ACK:
			if (ack(th->flags)) {
				fprintf(fi,"Server: connection closed\n");
				tcps = TCS_CLOSED;
			}
		break;
	}


	fprintf(fi,"\n");
}


